import XCTest
@testable import death_app_Watch_App

/// Unit tests for medical disclaimer functionality
/// Ensures App Store compliance and proper user consent tracking
class DisclaimerTests: XCTestCase {
    
    var disclaimerService: DisclaimerService!
    
    override func setUp() {
        super.setUp()
        disclaimerService = DisclaimerService()
        // Clean up UserDefaults for testing
        disclaimerService.resetDisclaimerAcceptance()
    }
    
    override func tearDown() {
        disclaimerService.resetDisclaimerAcceptance()
        disclaimerService = nil
        super.tearDown()
    }
    
    func testInitialDisclaimerState() {
        // Test that new installations require disclaimer acceptance
        XCTAssertFalse(disclaimerService.hasAcceptedDisclaimer, "New installations should require disclaimer acceptance")
        XCTAssertTrue(disclaimerService.shouldShowDisclaimer, "Disclaimer should be shown for new installations")
    }
    
    func testDisclaimerAcceptance() {
        // Test disclaimer acceptance flow
        XCTAssertFalse(disclaimerService.hasAcceptedDisclaimer)
        
        disclaimerService.acceptDisclaimer()
        
        XCTAssertTrue(disclaimerService.hasAcceptedDisclaimer, "Disclaimer should be accepted after user consent")
        XCTAssertFalse(disclaimerService.shouldShowDisclaimer, "Disclaimer should not be shown after acceptance")
    }
    
    func testDisclaimerPersistence() {
        // Test that disclaimer acceptance persists across app launches
        disclaimerService.acceptDisclaimer()
        XCTAssertTrue(disclaimerService.hasAcceptedDisclaimer)
        
        // Simulate app restart by creating new service instance
        let newService = DisclaimerService()
        XCTAssertTrue(newService.hasAcceptedDisclaimer, "Disclaimer acceptance should persist across app launches")
    }
    
    func testDisclaimerReset() {
        // Test disclaimer reset functionality
        disclaimerService.acceptDisclaimer()
        XCTAssertTrue(disclaimerService.hasAcceptedDisclaimer)
        
        disclaimerService.resetDisclaimerAcceptance()
        XCTAssertFalse(disclaimerService.hasAcceptedDisclaimer, "Disclaimer should be reset")
        XCTAssertTrue(disclaimerService.shouldShowDisclaimer, "Disclaimer should be required after reset")
    }
    
    func testDisclaimerInfoTracking() {
        // Test that disclaimer acceptance is properly tracked
        disclaimerService.acceptDisclaimer()
        
        let info = disclaimerService.getDisclaimerInfo()
        
        XCTAssertTrue(info["hasAccepted"] as? Bool ?? false, "Should track acceptance")
        XCTAssertNotNil(info["acceptedDate"], "Should track acceptance date")
        XCTAssertEqual(info["disclaimerVersion"] as? String, "2.0", "Should track disclaimer version")
        XCTAssertFalse(info["needsReAcceptance"] as? Bool ?? true, "Should not need re-acceptance")
    }
    
    func testDisclaimerVersionUpdates() {
        // Test that version updates require re-acceptance
        disclaimerService.acceptDisclaimer()
        XCTAssertTrue(disclaimerService.hasAcceptedDisclaimer)
        
        // Simulate version update by manually setting old version
        UserDefaults.standard.set("1.0", forKey: "disclaimerVersion")
        
        // Create new service to simulate app update
        let updatedService = DisclaimerService()
        XCTAssertFalse(updatedService.hasAcceptedDisclaimer, "Should require re-acceptance after version update")
    }
    
    func testAppStoreCompliance() {
        // Test that disclaimer contains required elements for App Store compliance
        let disclaimerInfo = disclaimerService.getDisclaimerInfo()
        
        XCTAssertNotNil(disclaimerInfo["currentVersion"], "Should track disclaimer version for compliance")
        
        // Test that service properly initializes without crashing
        let newService = DisclaimerService()
        XCTAssertNotNil(newService, "Service should initialize properly for App Store compliance")
    }
}